package main

import (
	"flag"
	"log"
	"log/slog"
	"net/http"
	"time"

	"github.com/prometheus/client_golang/prometheus"
	"github.com/prometheus/client_golang/prometheus/collectors"
	"github.com/prometheus/client_golang/prometheus/promhttp"

	"git.majava.org/software/edif"
	"git.majava.org/software/edif/config"
	"git.majava.org/software/edif/ns"
)

type Edif struct {
	cfg      *config.Config
	registry *prometheus.Registry
}

func main() {
	configPath := flag.String("config-path", edif.DefaultConfigPath, "Configuration file location")
	debug := flag.Bool("debug", false, "Enable verbose logging")
	flag.Parse()

	cfg, err := config.Load(*configPath)
	if err != nil {
		log.Panicf("Failed to load config! %v", err)
	}

	if *debug {
		slog.SetLogLoggerLevel(slog.LevelDebug)
	}

	registry := prometheus.NewRegistry()
	registry.MustRegister(
		collectors.NewGoCollector(),
		collectors.NewProcessCollector(collectors.ProcessCollectorOpts{}),
	)

	app := Edif{
		cfg:      cfg,
		registry: registry,
	}

	go app.FetchNameserverData()
	app.Listen()
}

func (e *Edif) FetchNameserverData() {
	q := ns.NewQuerier(e.registry, e.cfg.Recursor)
	for {
		for i, zone := range e.cfg.Zones {
			slog.Debug("querying zone",
				"index", i, "ns", zone.Name)
			q.FetchData(zone)
		}
		time.Sleep(15 * time.Second)
	}
}

func (e *Edif) Listen() {
	http.Handle("/metrics", promhttp.HandlerFor(e.registry, promhttp.HandlerOpts{Registry: e.registry}))
	slog.Info("started server", "address", e.cfg.ListenOn)
	log.Fatal(http.ListenAndServe(e.cfg.ListenOn, nil))
}
