from os import readlink
from pathlib import Path


def ascii_transliterate(text: str) -> str:
    """Transliterate an arbitrary text to printable ASCII using '?' as a replacement character."""
    chars = []
    for c in text:
        if 19 < ord(c) < 128:
            chars.append(c)
        else:
            chars.append("?")

    return "".join(chars)


def get_validated_symlink(template_path: Path) -> Path:
    """Sanitize for simple symlinks to local non-dot files, return dest"""
    # pathlib doesn't do raw readlink, and there's no good way to implement
    # what we're checking here without it:
    dest = Path(readlink(str(template_path)))
    if str(dest) != dest.name:
        raise Exception(
            f"{template_path.name} -> {dest} : Link dest must stay in same dir"
        )

    # Ensure link is to a real file that exists (not double-indirection with
    # another link, not dangling, not to a special/dir, etc)
    tmpl_dest = template_path.parent / dest.name
    if tmpl_dest.is_symlink() or not tmpl_dest.is_file():
        raise Exception(
            f"{template_path.name} -> {dest.name} : Link must be to an existing real file"
        )
    return tmpl_dest
