import logging
import subprocess
from collections.abc import Generator
from enum import IntEnum, StrEnum
from pathlib import Path


LOGGER = logging.getLogger(__name__)


class KeyType(StrEnum):
    KSK = "KSK"
    ZSK = "ZSK"


class Algorithm(IntEnum):
    # https://en.wikipedia.org/wiki/Domain_Name_System_Security_Extensions#Algorithms
    ED25519 = 15


class ZoneSigner:
    def __init__(
        self,
        zone_name: str,
        directory: Path,
    ) -> None:
        self.zone_name = zone_name
        self.directory = directory

    def generate_key(
        self, key_type: KeyType, algorithm: Algorithm = Algorithm.ED25519
    ) -> None:
        LOGGER.info(
            "Generating %s %s DNSSEC key for %s",
            key_type.name,
            algorithm.name,
            self.zone_name,
        )

        subprocess.check_output(
            [
                "/usr/bin/dnssec-keygen",
                "-K",
                str(self.directory),
                "-f",
                key_type.value,
                "-a",
                str(algorithm.value),
                self.zone_name,
            ]
        )

    def get_key_files(self) -> Generator[Path]:
        return self.directory.glob("*.key")

    def generate_and_get_keys(self) -> Generator[Path]:
        self.ensure_keys()
        return self.get_key_files()

    def has_keys(self) -> bool:
        return self.directory.exists() and any(self.get_key_files())

    def ensure_keys(self) -> None:
        if not self.directory.exists():
            self.directory.mkdir()

        if self.has_keys():
            LOGGER.debug("Zone %s has existing DNSSEC keys", self.zone_name)
            # TODO: key rotation?
            return

        self.generate_key(key_type=KeyType.ZSK)
        self.generate_key(key_type=KeyType.KSK)

    def sign_zone_file(self, zone_unsigned: Path) -> Path:
        keys = [str(key) for key in self.get_key_files()]

        subprocess.check_output(
            [
                "/usr/bin/dnssec-signzone",
                "-K",
                str(self.directory),
                "-o",
                self.zone_name,
                str(zone_unsigned),
                *keys,
            ],
            cwd=str(self.directory),
        )

        return zone_unsigned.parent / f"{zone_unsigned.name}.signed"


class Dnssec:
    def __init__(self, base_dir: Path) -> None:
        self.zones: dict[str, ZoneSigner] = {}
        self.base_dir = base_dir

    def get_zone(self, name: str) -> ZoneSigner:
        name_nodot = name.removesuffix(".")
        if name_nodot not in self.zones:
            self.zones[name_nodot] = ZoneSigner(
                name_nodot,
                self.base_dir / name_nodot,
            )

        return self.zones[name_nodot]
